/* ---------------------------------------------------------------------------
    2019 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */

package pkcs;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

// IAIK wrapper
import iaik.pkcs.pkcs11.wrapper.*;

import ui.MainWindow;
import ui.ScenarioInterface;
import ui.MainWindow.LogType;


public class Scenarios implements ScenarioInterface
{	
	/*
	
	*/
	public static String hidProductName = "ActivClient";
	public static String hidCopyrightYear = "2019";
	
	/*
	 * Original text to be signed is: Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.
	 * Compute the digest of this message using the API of your choice. MD5 for the given text above is: 818c6e601a24f72750da0f6c9b8ebe28
	 */ 
	private byte[] m_dataToSign;
	private byte[] m_signedData;
	private byte[] m_unlockCode;
	
	/**
	* PKCS#11 instance. It is bound to a third party PKCS service provider, which is a DLL under Windows.
	* Here, only the ActivIdentity PKCS#11 service provider is used.
	*/
	private PKCS11	m_pkcsInst;
	
	/**
	* Path to the PKCS service provider to use.
	*/
	private String m_sPkcsSP = "acpkcs211.dll";
	/**
	* Absolute path to the PKCS wrapper to use. Used when linking to the IAIK wrapper.
	*/
	private String m_sPkcsWrapper = "C:\\Program Files\\HID Global\\ActivClient\\pkcs11wrapper.dll";
	
	
	private byte [] m_dataToEncrypt = {(byte)0x01, (byte)0x02, (byte)0x03, (byte)0x04};
	
	private long m_sessionHandle, m_unlockObjHandle;
	private long m_privateKeyHandle, m_publicKeyHandle, m_secretKeyHandle, m_otpKeyHandle;
	private final String m_secretKeyLabel = "label";
	private byte [] m_encryptedData;
	
	private String [] m_readerList = null;
	private String [] m_certList = null;
	private long [] m_slotsIDList = null;
	private long m_selectedSlot = 0; 
	private long m_selectedCertificate = 0;
	private long [] m_certHandleList = null;
	private byte [] m_selectedCertID = null;
	private boolean m_privateKeyIsPINAlways;
	private boolean m_cardCanBeUnblocked;
	
	private CK_TOKEN_INFO m_tokenInfo = null;
	private CK_MECHANISM m_mechanism = null;
	
	private MainWindow m_mainWindow;

	private boolean useUtf8 = true;
	
	public Scenarios()
	{
		try
		{
			m_dataToSign = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.".getBytes("UTF8");
		}
		catch (UnsupportedEncodingException e)
		{
			m_dataToSign = null;
			e.printStackTrace();
			System.exit(-1);
		}
		
		m_mainWindow = new MainWindow(this, new String[]{"Data signature", "Login", "Secret key usage", "OTP generation"}, "HID Global PKCS API Java sample for IAIK");
		
		m_mainWindow.print("------------------------------------------------------------------------------------");
    	m_mainWindow.print("HID Global PKCS#11 API Java Sample for IAIK");
    	m_mainWindow.print("Copyright  " + hidCopyrightYear + " HID Global Corporation/ASSA ABLOY AB. All rights reserved.");
    	m_mainWindow.print("------------------------------------------------------------------------------------");
		
		// Initializes the Cryptoki library
		try
		{
			//m_pkcsInst = PKCS11.getInstance(m_sPkcsSP, "C_GetFunctionList", null, false);
			// With the IAIK wrapper
			m_pkcsInst = PKCS11Connector.connectToPKCS11Module(m_sPkcsSP, m_sPkcsWrapper);
			m_pkcsInst.C_Initialize(null, useUtf8);			
		} 
		catch (IOException | PKCS11Exception ex)
		{
			m_mainWindow.print("Exception : " + ex.getMessage(), LogType.ScenarioError);
		}
		
		String apiVersion = null;
		try
		{
			CK_INFO info = m_pkcsInst.C_GetInfo();
			apiVersion = Integer.toString((int)info.cryptokiVersion.major) + "." + Integer.toString((int)info.cryptokiVersion.minor);
		}
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetInfo failed with error " + ex.getMessage(), LogType.APIError);
        }
		String popupMsg = 		"This sample demonstrates how to use the PKCS #11 API version \""+apiVersion+"\" provided by " + hidProductName + " from a Java program.\n"
							+	"It implements following scenarios:\n"
							+	"        -    How to use a signature key to sign some data.\n"
							+	"        -    How to login to the card.\n"
							+	"        -    How to write a secret key on the card, use it to encrypt or decrypt some data and then delete it.\n"
							+	"        -    How to generate an OTP.\n";
		m_mainWindow.showMessageDialog(popupMsg);
	}
	
	/**
    * Executes the steps of the login scenario.
    *
    */
	public void loginScenario()
	{
		m_mainWindow.print("Starting login scenario...", LogType.ScenarioBoundary);
		if(!preamble() || !openSession(false))
		{
			postamble(false, false);
			m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!getPINProperties())
		{
			postamble(true, false);
			m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if((m_tokenInfo.flags & PKCS11Constants.CKF_USER_PIN_LOCKED) == PKCS11Constants.CKF_USER_PIN_LOCKED)
		{
			if(!isCardCanBeUnblocked())
			{
				postamble(true, false);
				m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
				return;
			}
			if(m_cardCanBeUnblocked)
			{
				if(!getUnlockCode() || !unblockCard())
				{
					postamble(true, false);
					m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
					return;
				}
			}
		}
		else
		{
			if((m_tokenInfo.flags & PKCS11Constants.CKF_USER_PIN_TO_BE_CHANGED) == PKCS11Constants.CKF_USER_PIN_TO_BE_CHANGED)
			{
				if(!changePIN())
				{
					postamble(true, false);
					m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
					return;
				}
			}
			else
			{
				if(!logIntoCardApplication(false))
				{
					postamble(true, false);
					m_mainWindow.print("Login scenario aborted!", LogType.ScenarioBoundary);
					return;
				}
				else
				{
					m_mainWindow.print("Login successful.", LogType.ScenarioInfo);
				}
			}
		}
		postamble(true, true);
		m_mainWindow.print("Login scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the data signature scenario.
    *
    */
	public void dataSignatureScenario()
	{
		m_mainWindow.print("Starting Data signature scenario...", LogType.ScenarioBoundary);
		if(!preamble() || !openSession(false))
		{
			postamble(false, false);
			m_mainWindow.print("Data signature scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!getSelectedCertificate() || !getCertificateID() || !logIntoCardApplication(false))
		{
			postamble(true, false);
			m_mainWindow.print("Data signature scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!findPrivateKey() || !signData() || !findPublicKey() || !verifySignature())
		{
			postamble(true, true);
			m_mainWindow.print("Data signature scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		postamble(true, true);
		m_mainWindow.print("Data signature scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the OTP generation scenario.
    *
    */
	public void otpGenerationScenario()
	{
		m_mainWindow.print("Starting OTP generation scenario...", LogType.ScenarioBoundary);
		if(!preamble() || !openSession(false))
		{
			postamble(false, false);
			m_mainWindow.print("OTP generation scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!logIntoCardApplication(false))
		{
			postamble(true, false);
			m_mainWindow.print("OTP generation scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!findOTPKey() || !checkOTPGenerationType() || !getOTP())
		{
			postamble(true, true);
			m_mainWindow.print("OTP generation scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		postamble(true, true);
		m_mainWindow.print("OTP generation scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the secret key usage scenario.
    *
    */
	public void secretKeyUsageScenario()
	{
		m_mainWindow.print("Starting Secret key usage scenario...", LogType.ScenarioBoundary);
		if(!preamble() || !openSession(true))
		{
			postamble(false, false);
			m_mainWindow.print("Secret key usage scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!createObject() || !logIntoCardApplication(false))
		{
			postamble(true, false);
			m_mainWindow.print("Secret key usage scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!findSecretKey() || !encrypt() || !decrypt() || !logoutOfCardApplication())
		{
			postamble(true, true);
			m_mainWindow.print("Secret key usage scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!logIntoCardApplication(false))
		{
			postamble(true, false);
			m_mainWindow.print("Secret key usage scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!destroyObject())
		{
			postamble(true, true);
			m_mainWindow.print("Secret key usage scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		postamble(true, true);
		m_mainWindow.print("Secret key usage scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Ends a scenario by terminating the connection : log out of the card if needed, 
    * then disconnect from the reader.
    *
    */
	public void postamble(boolean sessionOpen, boolean loggedIn)
	{
		if(loggedIn)
		{
			logoutOfCardApplication();
		}
		if(sessionOpen)
		{
			disconnect();
		}
	}
	
	/**
    * Gets the PIN properties out of the current token.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean getPINProperties()
	{
		try
		{
			m_tokenInfo = m_pkcsInst.C_GetTokenInfo(m_selectedSlot);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetTokenInfo failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}		
		m_mainWindow.print("Min PIN length : " + m_tokenInfo.ulMinPinLen);
		m_mainWindow.print("Max PIN length : " + m_tokenInfo.ulMaxPinLen);
		return true;
	}
	
	/**
    * Checks whether the card can be unblocked.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean isCardCanBeUnblocked()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[1];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_CLASS;

		// From aipkcs.h : 
		// #define CKO_ACTI                  (CKO_VENDOR_DEFINED + 0x800) /**< HID Global Objects id starting index */
		// #define CKO_UNBLOCK_PIN           (CKO_ACTI + 10) /**< Unlock PIN object. */
		template[0].pValue = PKCS11Constants.CKO_VENDOR_DEFINED + 0x800 + 10;
		long [] objectsHandles = null;
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			objectsHandles = m_pkcsInst.C_FindObjects(m_sessionHandle, 1);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		// If something was found, the card can be unlocked.
		if(objectsHandles.length > 0)
		{
			m_unlockObjHandle = objectsHandles[0];
			m_cardCanBeUnblocked = true;
		}
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
	
	/**
    * Gets the PIN properties out of the current token.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean getUnlockCode()
	{
		String unlockCode = null;
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[1];
		template[0] = new CK_ATTRIBUTE();
		// From aipkcs.h : 
		// #define CKA_UNBLOCK_PIN_CHALLENGE_REQUIREMENT			(CKA_VENDOR_DEFINED + 0x340)
		template[0].type = PKCS11Constants.CKA_VENDOR_DEFINED + 0x340;		
		try
		{
			m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_unlockObjHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		byte[] challengeRequirement = (byte[])template[0].pValue;
		switch(challengeRequirement[0])
		{
		// CK_OTP_PARAM_IGNORED
		case 0x0:
			unlockCode = m_mainWindow.showPINDialog("Enter the unlock code");
	        if (unlockCode == null)
	        {
	        	m_mainWindow.print("Operation canceled");
	        	return false;
	        }
	        m_mechanism = new CK_MECHANISM();
	        // CKM_UNBLOCK_PIN_STATIC
	        m_mechanism.mechanism = PKCS11Constants.CKM_VENDOR_DEFINED + 0x801;
			break;
		// CK_OTP_PARAM_MANDATORY
		case 0x2:
			// CKA_UNBLOCK_PIN_CHALLENGE
			template[0].type = PKCS11Constants.CKA_VENDOR_DEFINED + 0x340 + 1;
			try
			{
				m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_unlockObjHandle, template, useUtf8);
			} 
			catch (PKCS11Exception ex)
			{
				m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
				return false;
			}
			m_mainWindow.print("Challenge = " + Helper.byteArrayToASCIIString((byte[])template[0].pValue));
			unlockCode = m_mainWindow.showPINDialog("Enter the challenge response");
	        if (unlockCode == null)
	        {
	        	m_mainWindow.print("Operation canceled");
	        	return false;
	        }
	        m_mechanism = new CK_MECHANISM();
			// CKM_UNBLOCK_PIN_DYNAMIC
			m_mechanism.mechanism = PKCS11Constants.CKM_VENDOR_DEFINED + 0x801 + 1;
			break;
		default:
			m_mainWindow.print("Unknown Challenge Requirement value.", LogType.ScenarioError);
			return false;
		}
		unlockCode = unlockCode.replace("-", "");
        m_unlockCode = Helper.convAsciiHexToBin(unlockCode);
        if(m_unlockCode == null)
        {
        	return false;
        }
		return true;
	}
	
	/**
    * Unlocks the card.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean unblockCard()
	{
		String newPIN = m_mainWindow.showPINDialog("Enter new PIN");
		String confirmPIN = m_mainWindow.showPINDialog("Confirm new PIN");
		if(!newPIN.equals(confirmPIN))
		{
			m_mainWindow.print("PINs do not match.", LogType.ScenarioError);
			return false;
		}
		
		/*
		 * The following features are not supported by this sample. This is because C_VerifyUnblockPINInit
		 * and C_VerifyUnblockPIN are proprietary functions belonging to HID Global, and thus are not 
		 * present in the PKCS standard. As a consequence, the IAIK wrapper and the Sun wrapper do not support
		 * these functions. 
		 * In order to use these functions in Java, one should create a custom JNI wrapper containing them.
		 * The following commented lines demonstrate their use as if they were present in the JNI wrapper.
		 * For a functioning example, see the equivalent C and C# samples. 
		 */
		m_mainWindow.print("Unsupported features. A custom wrapper is needed.", LogType.ScenarioError);
		/*
		try
		{
			m_pkcsInst.C_VerifyUnblockPINInit(m_sessionHandle, m_mechanism, m_unlockObjHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_VerifyUnblockPINInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_pkcsInst.C_VerifyUnblockPIN(m_sessionHandle, newPIN.toCharArray(), m_unlockCode);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_VerifyUnblockPIN failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		*/
		
		// Verify that the new PIN is valid
		try
		{
			m_pkcsInst.C_Login(m_sessionHandle, PKCS11Constants.CKU_USER, newPIN.toCharArray(), useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_Login failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		
		return true;
	}
	
	/**
    * Demonstrates the modification of the PIN.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean changePIN()
	{
		String oldPIN = m_mainWindow.showPINDialog("Enter your PIN");
		String newPIN = m_mainWindow.showPINDialog("Enter new PIN");
		String confirmPIN = m_mainWindow.showPINDialog("Confirm new PIN");
		if(!newPIN.equals(confirmPIN))
		{
			m_mainWindow.print("PINs do not match.", LogType.ScenarioError);
			return false;
		}
		
		/*
		try
		{
			m_pkcsInst.C_SetPIN(m_sessionHandle, oldPIN.toCharArray(), newPIN.toCharArray(), useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_SetPIN failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_pkcsInst.C_Login(m_sessionHandle, PKCS11Constants.CKU_USER, newPIN.toCharArray(), useUtf8);
		}
        catch (PKCS11Exception ex)
		{
        	m_mainWindow.print("C_Login failed with error " + ex.getMessage(), LogType.APIError);
        	return false;
		}
		*/
		
		return true;
	}

	/**
    * Gets the list of smart card readers (by calling getReaderList), then
    * prompts the user with a popup letting him choose the reader to use.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean preamble()
	{
		listReaders();
		if(m_readerList == null || m_readerList.length == 0)
		{
			m_mainWindow.print("No card reader with a card inserted was detected.", LogType.ScenarioInfo);
			return false;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the reader",  m_readerList);
		if(choice == -1)
		{
			return false;
		}
		m_selectedSlot = m_slotsIDList[choice];
		return true;
	}
	
	public boolean findPrivateKey()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[4];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_TOKEN;
		template[0].pValue = true;
		template[1] = new CK_ATTRIBUTE();
		template[1].type = PKCS11Constants.CKA_PRIVATE;
		template[1].pValue = true;
		template[2] = new CK_ATTRIBUTE();
		template[2].type = PKCS11Constants.CKA_CLASS;
		template[2].pValue = PKCS11Constants.CKO_PRIVATE_KEY;
		template[3] = new CK_ATTRIBUTE();
		template[3].type = PKCS11Constants.CKA_ID;
		template[3].pValue = m_selectedCertID;
		long [] objectsHandles = null;
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			objectsHandles = m_pkcsInst.C_FindObjects(m_sessionHandle, 1);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		if(objectsHandles.length == 0)
		{
			m_mainWindow.print("Private key not found.", LogType.ScenarioError);
			return false;
		}
		m_privateKeyHandle = objectsHandles[0];
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}

	public boolean findSecretKey()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[1];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_LABEL;
		template[0].pValue = m_secretKeyLabel;
		long [] objectsHandles = null;
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			objectsHandles = m_pkcsInst.C_FindObjects(m_sessionHandle, 1);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		if(objectsHandles.length == 0)
		{
			m_mainWindow.print("Secret key not found.", LogType.ScenarioError);
			return false;
		}
		m_secretKeyHandle = objectsHandles[0];
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
	
	public boolean checkOTPGenerationType()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[1];
		template[0] = new CK_ATTRIBUTE();
		// #define CKA_OTP_CHALLENGE_REQUIREMENT 0x00000224
		template[0].type = 0x00000224;
		try
		{
			m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_otpKeyHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		byte[] res = (byte[])template[0].pValue;
		// #define CK_OTP_PARAM_IGNORED       0
		if(res[0] != 0)
		{
			m_mainWindow.print("Only synchronous OTP generation is supported by this sample.", LogType.ScenarioError);
			return false;
		}
		
		return true;
	}
	
	public boolean findPublicKey()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[4];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_TOKEN;
		template[0].pValue = true;
		template[1] = new CK_ATTRIBUTE();
		template[1].type = PKCS11Constants.CKA_VERIFY;
		template[1].pValue = true;
		template[2] = new CK_ATTRIBUTE();
		template[2].type = PKCS11Constants.CKA_CLASS;
		template[2].pValue = PKCS11Constants.CKO_PUBLIC_KEY;
		template[3] = new CK_ATTRIBUTE();
		template[3].type = PKCS11Constants.CKA_ID;
		template[3].pValue = m_selectedCertID;
		long [] objectsHandles = null;
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			objectsHandles = m_pkcsInst.C_FindObjects(m_sessionHandle, 1);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		if(objectsHandles.length == 0)
		{
			m_mainWindow.print("Public key not found.", LogType.ScenarioError);
			return false;
		}
		m_publicKeyHandle = objectsHandles[0];
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
	
	public boolean findOTPKey()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[2];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_CLASS;
		// CKO_OTP_KEY is new for PKCS #11 v2.20 amendment 1 
		//#define CKO_OTP_KEY           0x00000008
		template[0].pValue = 0x8;
		template[1] = new CK_ATTRIBUTE();
		template[1].type = PKCS11Constants.CKA_TOKEN;
		template[1].pValue = true;
		long [] objectsHandles = null;
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			objectsHandles = m_pkcsInst.C_FindObjects(m_sessionHandle, 1);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		if(objectsHandles.length == 0)
		{
			m_mainWindow.print("No OTP objects were found on the card.", LogType.ScenarioError);
			return false;
		}
		m_otpKeyHandle = objectsHandles[0];
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
	
	public boolean encrypt()
	{
		CK_MECHANISM mechanism = new CK_MECHANISM();
		byte [] iv = {(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
		mechanism.mechanism = PKCS11Constants.CKM_DES3_CBC_PAD;
		mechanism.pParameter = iv;
		
		try
		{
			m_pkcsInst.C_EncryptInit(m_sessionHandle, mechanism, m_secretKeyHandle, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_EncryptInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_encryptedData = m_pkcsInst.C_Encrypt(m_sessionHandle, m_dataToEncrypt);
			// With the IAIK wrapper
			//m_encryptedData = new byte[8];
			//m_pkcsInst.C_Encrypt(m_sessionHandle, m_dataToEncrypt, 0, m_dataToEncrypt.length, m_encryptedData, 0, m_encryptedData.length);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_Encrypt failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		m_mainWindow.print("Encrypted data:", LogType.ScenarioInfo);
		m_mainWindow.print(Helper.byteArrayToASCIIString(m_encryptedData), LogType.APIInfo);
		
		return true;
	}
	
	public boolean decrypt()
	{
		CK_MECHANISM mechanism = new CK_MECHANISM();
		byte [] iv = {(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
		mechanism.mechanism = PKCS11Constants.CKM_DES3_CBC_PAD;
		mechanism.pParameter = iv;
		
		byte [] decryptedData;
		
		try
		{
			m_pkcsInst.C_DecryptInit(m_sessionHandle, mechanism, m_secretKeyHandle, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_DecryptInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			//decryptedData = new byte[4];
			//m_pkcsInst.C_Decrypt(m_sessionHandle, m_encryptedData, 0, m_encryptedData.length, decryptedData, 0, decryptedData.length);
			// With the IAIK wrapper
			decryptedData = m_pkcsInst.C_Decrypt(m_sessionHandle, m_encryptedData);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_Decrypt failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		m_mainWindow.print("Decrypted data:", LogType.ScenarioInfo);
		m_mainWindow.print(Helper.byteArrayToASCIIString(decryptedData), LogType.APIInfo);
		
		if(!Helper.compareArrays(decryptedData, m_dataToEncrypt))
		{
			m_mainWindow.print("Decrypted data and initial data are not equal!", LogType.ScenarioError);
			return false;
		}
		
		return true;
	}
	
	/**
    * Reads on the card to get the list of certificates (by calling checkCertificates), then
    * prompts the user with a popup letting him choose the certificate that will be used to sign.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean getSelectedCertificate()
	{
		if(!listCertificates())
		{
			return false;
		}
		if(m_certList == null || m_certList.length == 0)
		{
			m_mainWindow.print("No certificate found on the card.");
			return false;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the certificate",  m_certList);
		if(choice == -1)
		{
			return false;
		}
		m_selectedCertificate = m_certHandleList[choice];
		return true;
	}
	
	/**
    * Creates a secret key object on the card.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean createObject()
	{
		if(!logIntoCardApplication(false))
		{
			return false;
		}
		byte [] key3DES = {
			 (byte)0x01, (byte)0x02, (byte)0x03, (byte)0x04, (byte)0x05, (byte)0x06, (byte)0x07, (byte)0x08, 
			 (byte)0x11, (byte)0x12, (byte)0x13, (byte)0x14, (byte)0x15, (byte)0x16, (byte)0x17, (byte)0x18, 
			 (byte)0x21, (byte)0x22, (byte)0x23, (byte)0x24, (byte)0x25, (byte)0x26, (byte)0x27, (byte)0x28};
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[8];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_TOKEN;
		template[0].pValue = true;
		template[1] = new CK_ATTRIBUTE();
		template[1].type = PKCS11Constants.CKA_PRIVATE;
		template[1].pValue = true;
		template[2] = new CK_ATTRIBUTE();
		template[2].type = PKCS11Constants.CKA_CLASS;
		template[2].pValue = PKCS11Constants.CKO_SECRET_KEY;
		template[3] = new CK_ATTRIBUTE();
		template[3].type = PKCS11Constants.CKA_LABEL;
		template[3].pValue = m_secretKeyLabel;
		template[4] = new CK_ATTRIBUTE();
		template[4].type = PKCS11Constants.CKA_KEY_TYPE;
		template[4].pValue = PKCS11Constants.CKK_DES3;
		template[5] = new CK_ATTRIBUTE();
		template[5].type = PKCS11Constants.CKA_WRAP;
		template[5].pValue = (byte)0x1;
		template[6] = new CK_ATTRIBUTE();
		template[6].type = PKCS11Constants.CKA_VALUE;
		template[6].pValue = key3DES;
		template[7] = new CK_ATTRIBUTE();
		template[7].type = PKCS11Constants.CKA_VALUE_LEN;
		template[7].pValue = key3DES.length;
		
		try
		{
			m_secretKeyHandle = m_pkcsInst.C_CreateObject(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_CreateObject failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		if(!logoutOfCardApplication())
		{
			return false;
		}
		
		return true;
	}
	
	/**
    * Destroys the secret key object previously created on the card.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean destroyObject()
	{
		try
		{
			m_pkcsInst.C_DestroyObject(m_sessionHandle, m_secretKeyHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_DestroyObject failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
	
	public boolean getCertificateID()
	{
		CK_ATTRIBUTE[] certLabelTemplate = new CK_ATTRIBUTE[1];
		certLabelTemplate[0] = new CK_ATTRIBUTE();
		certLabelTemplate[0].type = PKCS11Constants.CKA_ID;
		try
		{
			m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_selectedCertificate, certLabelTemplate, useUtf8);
			m_selectedCertID = (byte[])certLabelTemplate[0].pValue;
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		return true;
	}
	
	public boolean getOTP()
	{
		CK_MECHANISM mechanism = new CK_MECHANISM();
		// #define CKM_ACTI            0x000002A0
		mechanism.mechanism = 0x000002A0;
		
		byte[] otp;
		
		try
		{
			m_pkcsInst.C_SignInit(m_sessionHandle, mechanism, m_otpKeyHandle, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_SignInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			// With the Sun PKCS wrapper and the CKA_ALWAYS_AUTHENTICATE flag set, this call will fail.
			// This is because the Sun wrapper internally calls C_Sign twice.
			otp = m_pkcsInst.C_Sign(m_sessionHandle, null);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_Sign failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		m_mainWindow.print("Generated OTP:");
		m_mainWindow.print(Helper.byteArrayToASCIIString(otp), LogType.APIInfo);

		return true;
	}
	
	/**
    * Gets the list of available smart card readers.
    *
    */
	public void listReaders()
	{
		try
		{
			m_slotsIDList = m_pkcsInst.C_GetSlotList(true);
			m_readerList = new String[m_slotsIDList.length];
			for(int i = 0; i < m_slotsIDList.length; i++)
			{
        		CK_SLOT_INFO tokenInfo = m_pkcsInst.C_GetSlotInfo(m_slotsIDList[i]);
        		m_readerList[i] = new String(tokenInfo.slotDescription);
    		}
		}
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetSlotInfo failed with error " + ex.getMessage(), LogType.APIError);
        }
	}
	
	/**
    * Open the PKCS session.
    *
    * @param slotID		the identifier of the slot to connect to
    * @param readWrite 	True if Open Session in Read/Write mode
    */
	public boolean openSession(boolean readWrite)
	{
		long flags;
		if(readWrite)
		{
			flags = PKCS11Constants.CKF_SERIAL_SESSION | PKCS11Constants.CKF_RW_SESSION;
		}
		else
		{
			flags = PKCS11Constants.CKF_SERIAL_SESSION;
		}
		try
		{
			m_sessionHandle = m_pkcsInst.C_OpenSession(m_selectedSlot, flags, null, null);
		}
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_OpenSession failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		return true;
	}

	/**
    * Logs out of the card
    *
    */
	public boolean logoutOfCardApplication()
	{
		try
		{
			m_pkcsInst.C_Logout(m_sessionHandle);
		}
        catch (PKCS11Exception ex)
		{
        	m_mainWindow.print("C_Logout failed with error " + ex.getMessage(), LogType.APIError);
        	return false;
		}
		return true;
	}
	
	/**
    * Disconnects from the selected card reader.
    *
    */
	public void disconnect()
	{
		try
		{
			m_pkcsInst.C_CloseSession(m_sessionHandle);
		}
		catch (PKCS11Exception e)
		{
			e.printStackTrace();
		}
	}
	
	/**
    * Uses the selected certificate to sign some data
    *
    * @param KeyReference	the reference key of the certificate
    * @param algorithmIdentifier	the identifier of the algorithm used to sign
    */
	public boolean signData()
    {
		CK_MECHANISM mechanism = new CK_MECHANISM();
		mechanism.mechanism = PKCS11Constants.CKM_MD5_RSA_PKCS;
		
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[1];
		template[0] = new CK_ATTRIBUTE();
		
		// #define CKA_ALWAYS_AUTHENTICATE  0x00000202
		// template[0].type = 0x202;
		// With the IAIK wrapper, we can directly write :
		template[0].type = PKCS11Constants.CKA_ALWAYS_AUTHENTICATE;
		
		try
		{
			m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_privateKeyHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		//byte [] result = (byte[])template[0].pValue;
		//m_privateKeyIsPINAlways = (result[0] == 0x1);
		// With the IAIK wrapper, we can directly write :
		m_privateKeyIsPINAlways = (boolean)template[0].pValue;
		
		for(int i = 0; i < 2; i++)
		{
			try
			{
				m_pkcsInst.C_SignInit(m_sessionHandle, mechanism, m_privateKeyHandle, useUtf8);
			} 
			catch (PKCS11Exception ex)
			{
				m_mainWindow.print("C_SignInit failed with error " + ex.getMessage(), LogType.APIError);
				return false;
			}			
			
			if(m_privateKeyIsPINAlways && !logIntoCardApplication(true))
			{
				return false;
			}
			
			try
			{
				// With the Sun PKCS wrapper and the CKA_ALWAYS_AUTHENTICATE flag set, this call will fail.
				// This is because the Sun wrapper internally calls C_Sign twice.
				m_signedData = m_pkcsInst.C_Sign(m_sessionHandle, m_dataToSign);
			} 
			catch (PKCS11Exception ex)
			{
				m_mainWindow.print("C_Sign failed with error " + ex.getMessage(), LogType.APIError);
				return false;
			}
			
			m_mainWindow.print("Signed data:");
			m_mainWindow.print(Helper.byteArrayToASCIIString(m_signedData), LogType.APIInfo);
			
		}
		return true;
    }
	
	/**
    * Fills the list of certificates present on the card.
    *
    */
	public boolean listCertificates()
	{
		CK_ATTRIBUTE[] template = new CK_ATTRIBUTE[3];
		template[0] = new CK_ATTRIBUTE();
		template[0].type = PKCS11Constants.CKA_CLASS;
		template[0].pValue = PKCS11Constants.CKO_CERTIFICATE;
		template[1] = new CK_ATTRIBUTE();
		template[1].type = PKCS11Constants.CKA_TOKEN;
		template[1].pValue = true;
		template[2] = new CK_ATTRIBUTE();
		//#define CKA_CERTIFICATE_CATEGORY        0x00000087
		template[2].type = 0x00000087;
		template[2].pValue = 1; // 1 = token user
		
		try
		{
			m_pkcsInst.C_FindObjectsInit(m_sessionHandle, template, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_certHandleList = m_pkcsInst.C_FindObjects(m_sessionHandle, 100);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjects failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		m_certList = new String[m_certHandleList.length];
		CK_ATTRIBUTE[] certLabelTemplate = new CK_ATTRIBUTE[1];
		certLabelTemplate[0] = new CK_ATTRIBUTE();
		certLabelTemplate[0].type = PKCS11Constants.CKA_LABEL;
		
		try
		{
			for(int i = 0; i < m_certHandleList.length; i++)
			{
				m_pkcsInst.C_GetAttributeValue(m_sessionHandle, m_certHandleList[i], certLabelTemplate, useUtf8);
				m_certList[i] = new String((char[])certLabelTemplate[0].pValue);
			}
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_GetAttributeValue failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_pkcsInst.C_FindObjectsFinal(m_sessionHandle);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_FindObjectsFinal failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		return true;
	}
	
	/**
    * Login to the card. This method asks the user for the PIN of the card, 
    * then calls C_Login to authenticate him.
    *
    * @return boolean	false if the authentication failed, or the user canceled. 
    * 					If this method returns true, the caller must log out of the card
    * 					when the operations are finished. 
    */
	public boolean logIntoCardApplication(boolean contextSpecific)
	{
		String sPin = m_mainWindow.showPINDialog();
        if (sPin == null)
        {
        	m_mainWindow.print("Operation canceled");
        	return false;
        }
        long flag = contextSpecific ? 2 : PKCS11Constants.CKU_USER;
        try
		{
			m_pkcsInst.C_Login(m_sessionHandle, flag, sPin.toCharArray(), useUtf8);
		}
        catch (PKCS11Exception ex)
		{
        	m_mainWindow.print("C_Login failed with error " + ex.getMessage(), LogType.APIError);
        	return false;
		}
		return true;
	}

	@Override
	public void runScenario(int scenarioIdx)
	{
		switch(scenarioIdx)
		{
		case 0:
			dataSignatureScenario();
			break;
		case 1:
			loginScenario();
			break;
		case 2:
			secretKeyUsageScenario();
			break;
		case 3:
			otpGenerationScenario();
			break;
		default:
			m_mainWindow.print("Combo box state corrupt", LogType.ScenarioError);
			break;
		}
	}
	
	public boolean verifySignature()
	{
		CK_MECHANISM mechanism = new CK_MECHANISM();
		mechanism.mechanism = PKCS11Constants.CKM_MD5_RSA_PKCS;
		
		try
		{
			m_pkcsInst.C_VerifyInit(m_sessionHandle, mechanism, m_publicKeyHandle, useUtf8);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_VerifyInit failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		try
		{
			m_pkcsInst.C_Verify(m_sessionHandle, m_dataToSign, m_signedData);
		} 
		catch (PKCS11Exception ex)
		{
			m_mainWindow.print("C_Verify failed with error " + ex.getMessage(), LogType.APIError);
			return false;
		}
		
		return true;
	}
}
